<?php
namespace App\Http\Controllers;
use App\Models\Expense;
use App\Models\Category;
use App\Models\Setting;
use App\Http\Requests\StoreExpenseRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ExpenseController extends Controller
{
  public function index(Request $r){
    $q = Expense::with('category')->orderByDesc('date');

    if($r->filled('category_id')) $q->where('category_id',$r->category_id);
    if($r->filled('from')) $q->whereDate('date','>=',$r->from);
    if($r->filled('to')) $q->whereDate('date','<=',$r->to);
    if($r->filled('vendor')) $q->where('vendor','like','%'.$r->vendor.'%');

    $expenses = $q->paginate(25)->withQueryString();

    $totalBudget = (float) Setting::get('total_budget', 0);
    $spent = (float) Expense::sum('gross_amount');
    $remaining = $totalBudget - $spent;

    $perCategory = Category::withSum('expenses as spent','gross_amount')->get();

    // Purchase price & ancillary costs (NRW, no broker)
    $purchasePrice = (float) Setting::get('purchase_price', 160000);
    $grEStRate = (float) Setting::get('grunderwerbsteuer_rate', 6.5);
    $notaryRate = (float) Setting::get('notary_land_reg_rate', 2.0);
    $grESt = round($purchasePrice * $grEStRate / 100, 2);
    $notary = round($purchasePrice * $notaryRate / 100, 2);
    $ancillary = $grESt + $notary;
    $totalAcquisition = $purchasePrice + $ancillary;

    return view('expenses.index', compact(
      'expenses','perCategory','totalBudget','spent','remaining',
      'purchasePrice','grEStRate','notaryRate','grESt','notary','ancillary','totalAcquisition'
    ));
  }

  public function create(){
    $categories = Category::orderBy('name')->get();
    return view('expenses.create', compact('categories'));
  }

  public function store(StoreExpenseRequest $r){
    $data = $r->validated();
    if($r->hasFile('file')){
      $data['file_path'] = $r->file('file')->store('invoices','public');
    }
    Expense::create($data);
    return redirect()->route('expenses.index')->with('ok','Rechnung gespeichert');
  }

  public function destroy(Expense $expense){
    if($expense->file_path) Storage::disk('public')->delete($expense->file_path);
    $expense->delete();
    return back()->with('ok','Eintrag gelöscht');
  }
}
